﻿using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Collections.Generic;
using System.Linq;
using System.ServiceModel;
using System.Text;
using System.Threading.Tasks;
using VA.PPMS.Context;

namespace VA.PPMS.CRM.Plugins.Batch
{
    public class BatchDetailCreate : IPlugin
    {
        private ITracingService _tracingService;
        private IOrganizationService _service;

        private const string PluginName = "BatchDetailCreate";
        private const string ProcessDateAttribute = "ppms_validationcompletedate";

        public void Execute(IServiceProvider serviceProvider)
        {
            // Tracing service for debugging
            _tracingService = (ITracingService)serviceProvider.GetService(typeof(ITracingService));

            // Get execution context
            IPluginExecutionContext context = (IPluginExecutionContext)serviceProvider.GetService(typeof(IPluginExecutionContext));

            if (context.InputParameters.Contains("Target") && context.InputParameters["Target"] is Entity)
            {
                _tracingService.Trace("Begin");

                // Obtain the target entity from the input parameters.
                Entity entity = (Entity)context.InputParameters["Target"];

                // Verify target entity type
                if (entity.LogicalName != "ppms_batchdetail")
                    return;

                _tracingService.Trace("Entity found");

                // Get organization service reference
                IOrganizationServiceFactory serviceFactory = (IOrganizationServiceFactory)serviceProvider.GetService(typeof(IOrganizationServiceFactory));
                _service = serviceFactory.CreateOrganizationService(context.UserId);

                try
                {
                    _tracingService.Trace("Check state of provider");

                    //var detail = GetBatchDetail(entity.Id);
                    var providerRef = entity.GetAttributeValue<EntityReference>("ppms_provider");
                    if (providerRef != null)
                    {
                        var provider = GetProvider(_service, providerRef.Id);
                        if (provider != null)
                        {
                            IList<SetStateRequest> requests = new List<SetStateRequest>();

                            // If the provider status is Active or the provider state is Inactive, 
                            // the batch detail validation is complete
                            if (provider.StatusCode.Value == (int)Account_StatusCode.Active)
                            {
                                // Set batch detail as complete
                                requests = PpmsHelper.MarkBatchDetailAsComplete(provider, _service);
                            }
                            else if (provider.StateCode.Value == AccountState.Inactive)
                            {
                                // Set batch detail as Rejected
                                var statusReason = provider.StatusCode;
                                if (statusReason != null)
                                {
                                    switch (statusReason.Value)
                                    {
                                        case (int)Account_StatusCode.LEIEExclusion:
                                            _tracingService.Trace("Provider in Inactive Status");
                                            PpmsHelper.CreateBatchDetailLeieResult(entity.Id, _service);
                                            break;
                                        case (int)Account_StatusCode.NPICheckFailure:
                                            _tracingService.Trace("Provider in Inactive Status");
                                            PpmsHelper.CreateBatchDetailNpiCheckResult(entity.Id, _service);
                                            break;
                                        default:
                                            break;
                                    }
                                }

                                // Set batch detail as complete
                                requests = PpmsHelper.MarkBatchDetailAsComplete(provider, _service);
                            }
                            else
                            {
                                _tracingService.Trace("Target state not found");
                            }

                            // Execute requests
                            if (requests.Count > 0)
                            {
                                _tracingService.Trace("Execute requests {0}", requests.Count);
                                foreach (var request in requests)
                                {
                                    _service.Execute(request);
                                }
                            }
                        }
                        else
                        {
                            _tracingService.Trace("Unable to retrieve provider");
                        }
                    }
                    else
                    {
                        _tracingService.Trace("Provider not set");
                    }
                }
                catch (FaultException<OrganizationServiceFault> ex)
                {
                    _tracingService.Trace("Fault: {0}", ex.ToString());
                    throw new InvalidPluginExecutionException(String.Format("An error occurred in {0}.", PluginName), ex);
                }
                catch (Exception ex)
                {
                    _tracingService.Trace("Exception: {0}", ex.ToString());
                    throw;
                }
            }
            _tracingService.Trace("Done");
        }

        private Account GetProvider(IOrganizationService service, Guid providerId)
        {
            using (var context = new PpmsContext(service))
            {
                var account = context.AccountSet.FirstOrDefault(a => a.AccountId == providerId);
                context.LoadProperty(account, new Relationship("ppms_account_batchdetail_provider"));
                return account;
            }
        }

        private Entity GetBatchDetail(Guid batchDetailId)
        {
            return _service.Retrieve("ppms_batchdetail", batchDetailId, new ColumnSet(new string[] { "ppms_batchdetailid", "ppms_name", "ppms_batch", "ppms_provider", ProcessDateAttribute }));
        }
    }
}
